<?php
/*
* Plugin Name: Botastico Script Inserter
* Description: Allows you to effortlessly embed the Botastico script, using your specific app id, directly into the head section of the HTML document.
* Version: 1.0.5
* Author: Botastico
* Author URI: https://botasti.co/
* Icon URI: assets/images/botastico-logo.svg
* License: GPLv2 or later
*/

// Create a submenu page under the "Settings" menu in the WordPress admin dashboard
function add_script_inserter_to_menu() {
    add_submenu_page(
        'options-general.php',
        'Botastico Script Inserter',
        'Botastico Script Inserter',
        'manage_options',
        'botastico-script-inserter',
        'script_inserter_settings_page'
    );
}
add_action('admin_menu', 'add_script_inserter_to_menu');

// Register settings and fields
function botastico_script_inserter_settings_init() {
    register_setting('botastico_script_inserter_settings_group', 'botastico_app_id', [
        'sanitize_callback' => 'sanitize_text_field',
    ]);

    register_setting('botastico_script_inserter_settings_group', 'botastico_exclude_paths', [
        'sanitize_callback' => 'sanitize_text_field',
    ]);
}
add_action('admin_init', 'botastico_script_inserter_settings_init');

// Settings page callback
function script_inserter_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.'));
    }

    ?>
    <div class="wrap">
        <h1>Botastico Script Inserter</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('botastico_script_inserter_settings_group');
            do_settings_sections('botastico_script_inserter_settings_group');
            ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><label for="botastico_app_id">App ID</label></th>
                    <td>
                        <input type="text" id="botastico_app_id" name="botastico_app_id" value="<?php echo esc_attr(get_option('botastico_app_id')); ?>" placeholder="Enter your App ID..." required />
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="botastico_exclude_paths">Exclude Paths</label></th>
                    <td>
                        <input type="text" id="botastico_exclude_paths" name="botastico_exclude_paths" value="<?php echo esc_attr(get_option('botastico_exclude_paths')); ?>" placeholder="e.g., programs, about, contact" />
                        <p class="description">Enter comma-separated URL path fragments where the chatbot icon should not be displayed. If left empty, the chatbot will appear on all pages.</p>
                    </td>
                </tr>
            </table>
            <?php submit_button('Save Settings'); ?>
        </form>
    </div>
    <?php
}

function insert_botastico_script() {
    $app_id = get_option('botastico_app_id');
    $exclude_paths = get_option('botastico_exclude_paths', '');

    if (!empty($app_id)) {
        // Prepare botasticoAppSettings object
        $botastico_settings = [
            'appId' => $app_id,
        ];

        if (!empty($exclude_paths)) {
            $exclude_paths_array = array_map('trim', explode(',', $exclude_paths));
            $botastico_settings['excludePaths'] = $exclude_paths_array;
        }

        $botastico_settings_json = json_encode($botastico_settings);

        echo '<script>
            window.botasticoAppSettings = ' . $botastico_settings_json . ';
            (function() {
                var script = document.createElement("script");
                script.src = "https://chatapps.botasti.co/main.js";
                script.async = true;
                document.head.appendChild(script);
            })();
        </script>';
    }
}
add_action('wp_head', 'insert_botastico_script');

// Register deactivation hook
register_deactivation_hook(__FILE__, 'botastico_script_inserter_deactivate');

function botastico_script_inserter_deactivate() {
    // Remove the app ID and exclude paths from WordPress options
    delete_option('botastico_app_id');
    delete_option('botastico_exclude_paths');
}

// Display admin notices
function botastico_admin_notices() {
    // Check if settings were updated
    if (get_transient('botastico_settings_saved')) {
        echo '<div class="notice notice-success is-dismissible">
            <p>Settings saved successfully.</p>
        </div>';
        delete_transient('botastico_settings_saved');
    }

    // Check for error messages
    if ($errors = get_transient('botastico_settings_errors')) {
        foreach ($errors as $error) {
            echo '<div class="notice notice-error is-dismissible">
                <p>' . esc_html($error) . '</p>
            </div>';
        }
        delete_transient('botastico_settings_errors');
    }
}
add_action('admin_notices', 'botastico_admin_notices');

// Modify the settings form submission to use transients for notices
function botastico_handle_form_submission() {
    // Ensure we're on the Botastico Script Inserter settings page
    if (isset($_GET['page']) && $_GET['page'] === 'botastico-script-inserter') {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Verify nonce for security (ensure to add a nonce field in the form)
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'botastico_script_inserter_settings_group-options')) {
                wp_die(__('Nonce verification failed.', 'botastico-script-inserter'));
            }

            $errors = [];

            // Validate and sanitize App ID
            $app_id = isset($_POST['botastico_app_id']) ? sanitize_text_field($_POST['botastico_app_id']) : '';
            if (empty($app_id)) {
                $errors[] = 'App ID cannot be empty.';
            }

            // Validate and sanitize Exclude Paths
            $exclude_paths_input = isset($_POST['botastico_exclude_paths']) ? sanitize_text_field($_POST['botastico_exclude_paths']) : '';
            $exclude_paths_array = array_map('trim', explode(',', $exclude_paths_input));

            foreach ($exclude_paths_array as $path) {
                if (empty($path)) {
                    $errors[] = 'Exclude paths cannot contain empty entries.';
                    break;
                }
                if (!is_string($path)) {
                    $errors[] = 'Each exclude path must be a valid string.';
                    break;
                }
            }

            if (empty($errors)) {
                update_option('botastico_app_id', $app_id);
                update_option('botastico_exclude_paths', implode(', ', $exclude_paths_array));
                set_transient('botastico_settings_saved', true, 30);
            } else {
                set_transient('botastico_settings_errors', $errors, 30);
            }

            // Redirect to avoid form resubmission
            wp_redirect(add_query_arg('page', 'botastico-script-inserter', admin_url('options-general.php')));
            exit;
        }
    }
}
add_action('admin_init', 'botastico_handle_form_submission');
?>